<?php
ini_set('session.save_path', __DIR__ . '/tmp_sessions');
session_start();
require_once 'config/db.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: login.php');
    exit();
}

$vehicle_id = $_GET['id'] ?? null;
if (!$vehicle_id) {
    die("No vehicle selected.");
}

// Get vehicle info and assigned driver
$stmt = $pdo->prepare("SELECT v.*, u.username AS driver_name
                       FROM vehicles v
                       LEFT JOIN user_vehicles uv ON v.id = uv.vehicle_id
                       LEFT JOIN users u ON u.id = uv.user_id
                       WHERE v.id = ?");
$stmt->execute([$vehicle_id]);
$vehicle = $stmt->fetch();

if (!$vehicle) {
    die("Vehicle not found.");
}

// Get last 14 daily checks
$checks_stmt = $pdo->prepare("SELECT dc.*, u.username 
                              FROM daily_checks dc 
                              JOIN users u ON dc.driver_id = u.id
                              WHERE dc.vehicle_id = ?
                              ORDER BY dc.created_at DESC
                              LIMIT 14");
$checks_stmt->execute([$vehicle_id]);
$checks = $checks_stmt->fetchAll();

// Get service history
$service_stmt = $pdo->prepare("SELECT * FROM service_history WHERE vehicle_id = ? ORDER BY service_date DESC");
$service_stmt->execute([$vehicle_id]);
$services = $service_stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Vehicle Overview</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="container py-4">
  <h2>Vehicle Overview: <?= htmlspecialchars($vehicle['plate_number']) ?></h2>

  <div class="mb-4">
    <h4>Vehicle Information</h4>
    <table class="table table-bordered">
      <tr><th>Plate</th><td><?= htmlspecialchars($vehicle['plate_number']) ?></td></tr>
      <tr><th>Make</th><td><?= htmlspecialchars($vehicle['make']) ?></td></tr>
      <tr><th>Model</th><td><?= htmlspecialchars($vehicle['model']) ?></td></tr>
      <tr><th>Mileage</th><td><?= $vehicle['mileage'] ?></td></tr>
      <tr><th>Last Service</th><td><?= $vehicle['last_service'] ? date('d-m-Y', strtotime($vehicle['last_service'])) : 'N/A' ?></td></tr>
      <tr><th>Next Service Due</th><td><?= $vehicle['service_due'] ? date('d-m-Y', strtotime($vehicle['service_due'])) : 'N/A' ?></td></tr>
      <tr><th>Assigned Driver</th><td><?= htmlspecialchars($vehicle['driver_name'] ?? 'Unassigned') ?></td></tr>
    </table>
  </div>

  <div class="mb-4">
    <h4>Last 14 Daily Checks</h4>
    <table class="table table-striped table-bordered">
      <thead>
        <tr>
          <th>Date</th>
          <th>Time</th>
          <th>Driver</th>
          <th>Mileage</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($checks as $check): 
            $datetime = new DateTime($check['created_at']);
        ?>
        <tr>
          <td><?= $datetime->format('d-m-Y') ?></td>
          <td><?= $datetime->format('H:i') ?></td>
          <td><?= htmlspecialchars($check['username']) ?></td>
          <td><?= isset($check['mileage']) ? htmlspecialchars($check['mileage']) : "N/A" ?></td>
          <td>
            <a href="view_check.php?id=<?= $check['id'] ?>" class="btn btn-sm btn-info">View</a>
            <a href="edit_check.php?id=<?= $check['id'] ?>" class="btn btn-sm btn-warning">Edit</a>
          </td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="mb-4">
    <h4>Service History</h4>
    <table class="table table-striped table-bordered">
      <thead>
        <tr>
          <th>Date</th>
          <th>Mileage</th>
          <th>Service Type</th>
          <th>Performed By</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($services as $s): ?>
        <tr>
          <td><?= date('d-m-Y', strtotime($s['service_date'])) ?></td>
          <td><?= $s['mileage'] ?></td>
          <td><?= ucfirst($s['service_type']) ?></td>
          <td><?= htmlspecialchars($s['performed_by']) ?></td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</body>
</html>
